#if NUNIT

#region References

using System;
using System.Collections;
using System.Data;
using System.Text;

using NUnit.Framework;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;

using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using SPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;

using gov.va.med.vbecs.BOL.UnitTests;
using gov.va.med.vbecs.UnitTests;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	[TestFixture]
	public class Parsers_BceHL7Parser : BaseTest 
	{
		#region Variables

		private const string LONG_REACTIONS = "^~Yes^~*New Cough>><<IV Line Infiltrated>><<*Headache, Severe>><<*Some crazy symptom this patient has eh, this has to be really long to test NCover, not that fun but what can you do, sooooooooooooooooooooo yeahhhhhhhhhhhh errrrrrrrrr textttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttttt>><<Comment Here";
		private const string LONG_INTERRUPTION_COMMENT_BASE = "^~No^~*Anxiety>><<";
		private const string NO_NOTHIN = "^^";
		private const string NO_REACTIONS = "^~No^~*Anxiety>><<Comment Here";
		private const string REACTIONS = "^~Yes^~*New Cough>><<IV Line Infiltrated>><<*Headache, Severe>><<Comment Here";

		private Guid _issuedUnitGuid;
        //private int _lockFormId = -420;
        //private BOL.LockManager _lockManager;

		private static HL7Interface _hl7Interface;

		#endregion 

		#region Constructors

        ///// <summary>
        ///// LOHSE 3/21 CONFIG FIX?
        ///// </summary>
        //static Parsers_BceHL7Parser()
        //{
        //    _hl7Interface = new HL7Interface( InterfaceName.BCE_COTS.ToString() );
        //}

		#endregion

		#region Setup & TearDown

	    private bool doOnce = true;

		[SetUp]
		public void SetUp()
		{
		    this.BaseSetUp();

		    if (doOnce)
		    {
                _hl7Interface = new HL7Interface(InterfaceName.BCE_COTS.ToString());
		        doOnce = false;
		    }

	        Common.LogonUser.LogonUserDUZ = "53311";
			Common.LogonUser.LogonUserDivisionCode = UnitTestConfig.TestDivisionCode;
			//
			_issuedUnitGuid = DataCreator.CreateIssuedBloodUnit();
			Assert.IsTrue( _issuedUnitGuid != Guid.Empty, "IssuedUnit Created" );
		}

		[TearDown]
		public void TearDown()
		{
		}

		#endregion

		/// <summary>
		/// ParseHL7Message_Reactions_Pass - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_Reactions_Pass()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
			Assert.IsNotNull( protocolMessage, "Bce Message created" );
			//
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check");
		}

		/// <summary>
		/// ParseHL7Message_No_Reactions_Pass - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_No_Reactions_Pass()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, NO_REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
			Assert.IsNotNull( protocolMessage );
			//
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message_No_Reactions_Nothing_Pass - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_No_Reactions_Nothing_Pass()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, NO_NOTHIN, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
			Assert.IsNotNull( protocolMessage, "Bce Message Parsed" );
			//
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check");
		}

		/// <summary>
		/// ParseHL7Message_No_Reactions_Long_Reactions - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_No_Reactions_Long_Reactions()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, LONG_REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
			Assert.IsNotNull( protocolMessage, "Bce Message Parsed" );
			//
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message_No_Reactions_Long_Interruption_Comment - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_No_Reactions_Long_Interruption_Comment()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, LONG_INTERRUPTION_COMMENT_BASE + GetLongComment(520), false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
			//
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (interface is null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseHL7Message_Fail_Null_Interface()
		{
			string bceBtsMessage = GetBceBtsMessage( _issuedUnitGuid, NO_REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( null, bceBtsMessage );
			//
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
		}

		/// <summary>
		/// ParseHL7Message - FAIL (message is null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseHL7Message_Fail_Null_Message()
		{
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, null );
			//
			Assert.IsNotNull( protocolMessage, "Bce Message Parsed" );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (interface and message are null)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseHL7Message_Fail_Null_Interface_And_Message()
		{
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( null, null );
			//
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
		}

		/// <summary>
		/// ParseHL7Message - FAIL (interface and message are null)
		/// </summary>
		[Test]
		public void ParseHL7Message_Fail_Broken_Rules()
		{
			Guid issuedUnitGuid = DataCreator.CreateIssuedBloodUnit();
			Assert.IsTrue( issuedUnitGuid != Guid.Empty, "IssuedUnit Created" );
			//
			string bceBtsMessage = GetBceBtsMessage( issuedUnitGuid, NO_REACTIONS, true );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
			Assert.AreEqual("AR", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (locking)
		/// </summary>
		[Test]
		public void ParseHL7Message_Fail_Locking()
		{
			Guid issuedUnitGuid = DataCreator.CreateIssuedBloodUnit();
			Assert.IsTrue( issuedUnitGuid != Guid.Empty, "IssuedUnit Created" );
			//
			string bceBtsMessage = GetBceBtsMessage( issuedUnitGuid, NO_REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
			//
			string query = "SELECT TOP 1 * FROM IssuedUnit IU JOIN PatientTransfusion PT ON IU.BloodUnitGuid = PT.BloodUnitGuid " +
				"WHERE IU.IssuedUnitGuid = '" + issuedUnitGuid.ToString() + "'";
			//
			DataRow drPatientAndBloodUnitData = UnitTestUtil.RunSQLGetDataRow( query );
			Assert.IsNotNull( drPatientAndBloodUnitData, "Transfusion Check" );
			//
			Guid bloodUnitGuid = (Guid)drPatientAndBloodUnitData[TABLE.PatientTransfusion.BloodUnitGuid];
			Guid patientGuid = (Guid)drPatientAndBloodUnitData[TABLE.PatientTransfusion.PatientGuid];
			//
			DataTable dtPatientTransfusionData = UnitTestUtil.RunSQLGetDataTable( string.Concat("SELECT * FROM PatientTransfusion WHERE PatientGuid = '", patientGuid, "' AND BloodUnitGuid = '", bloodUnitGuid, "'") );
			//
			BOL.PatientTransfusion patientTransfusion = new gov.va.med.vbecs.BOL.PatientTransfusion( dtPatientTransfusionData.Rows[0] );
			//
			Guid bloodUnitStatusGuid = (Guid)DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(bloodUnitGuid).Rows[0][TABLE.BloodUnitStatus.BloodUnitStatusGuid];

            BOL.LockManager tmpLM = new BOL.LockManager(666, 666, BOL.LockManager.DefaultLockInactivityTimeoutMin);
		    tmpLM.LockFunction = LockFunctions.UnitTests;
			if ( patientTransfusion.IsNew )
			{
				// Need to lock blood unit record if new
			    tmpLM.LockRecord(bloodUnitStatusGuid, false);
			}
			else
			{
				// Also need to lock patient transfusion record if editing
                tmpLM.LockRecord(bloodUnitStatusGuid, false);
                tmpLM.LockRecord(patientTransfusion.PatientTransfusionGuid, false);
			}
            ////
			HL7ProtocolMessage message = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
            Assert.NotNull(message, "Bce Message Parsed");
			//
			Assert.AreEqual("AE", HL7Utility.ParseGetAckCode(message.GetMessage()), "Ack check");
			//
            tmpLM.UnlockAllRecordsForUseCaseUser(Common.LockFunctions.UnitTests,666,666);
		}

		/// <summary>
		/// ParseHL7Message_Fail_Locking_Max_Retry_Attemps - FAIL (locking)
		/// </summary>
		[Test]
		public void ParseHL7Message_Fail_Locking_Max_Retry_Attemps()
		{
			Guid issuedUnitGuid = DataCreator.CreateIssuedBloodUnit();
			Assert.IsTrue( issuedUnitGuid != Guid.Empty, "IssuedUnit Created");
			//
			string bceBtsMessage = GetBceBtsMessage( issuedUnitGuid, NO_REACTIONS, false );
			//
			HL7ProtocolMessage protocolMessage = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
            Assert.IsNotNull(protocolMessage, "Bce Message Parsed");
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(protocolMessage.GetMessage()), "Ack Check" );
			//
			string query = "SELECT TOP 1 * FROM IssuedUnit IU JOIN PatientTransfusion PT ON IU.BloodUnitGuid = PT.BloodUnitGuid " +
				"WHERE IU.IssuedUnitGuid = '" + issuedUnitGuid.ToString() + "'";
			//
			DataRow drPatientAndBloodUnitData = UnitTestUtil.RunSQLGetDataRow( query );
			Assert.IsNotNull( drPatientAndBloodUnitData, "Transfusion Data Check" );
			//
			Guid bloodUnitGuid = (Guid)drPatientAndBloodUnitData[TABLE.PatientTransfusion.BloodUnitGuid];
			Guid patientGuid = (Guid)drPatientAndBloodUnitData[TABLE.PatientTransfusion.PatientGuid];
			//
			DataTable dtPatientTransfusionData = UnitTestUtil.RunSQLGetDataTable( string.Concat("SELECT * FROM PatientTransfusion WHERE PatientGuid = '", patientGuid, "' AND BloodUnitGuid = '", bloodUnitGuid, "'") );
			//
			BOL.PatientTransfusion patientTransfusion = new gov.va.med.vbecs.BOL.PatientTransfusion( dtPatientTransfusionData.Rows[0] );
			//
			Guid bloodUnitStatusGuid = (Guid)DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(bloodUnitGuid).Rows[0][TABLE.BloodUnitStatus.BloodUnitStatusGuid];
			//
			if (!patientTransfusion.IsNew )
			{
                DAL.LockManager.LockRecord(patientTransfusion.PatientTransfusionGuid, 8008, Common.LockFunctions.UC069PostTransfusionInformation, false, 7734, BOL.LockManager.DefaultLockInactivityTimeoutMin);
			}
            DAL.LockManager.LockRecord(bloodUnitStatusGuid, 8008, Common.LockFunctions.UC069PostTransfusionInformation, false, 7734, BOL.LockManager.DefaultLockInactivityTimeoutMin);
			//
			_hl7Interface.ReTransmitAttempts = 1;
			HL7ProtocolMessage message = BceHL7Parser.ParseHL7Message( _hl7Interface, bceBtsMessage );
			//
            Assert.NotNull(message, "Bce Message Parsed");
			//
			Assert.AreEqual("AE", HL7Utility.ParseGetAckCode(message.GetMessage()), "Ack Check" );      //This means the records are locked

            //record locked + transmitattempts
            message = BceHL7Parser.ParseHL7Message(_hl7Interface, bceBtsMessage);
            Assert.NotNull(message, "Bce Message Parsed");
            Assert.AreEqual("AR", HL7Utility.ParseGetAckCode(message.GetMessage()), "Ack Check");      //This means the records are locked, and re-trasnmits were exceeded
		    
            BOL.LockManager.UnlockAllRecordsForUser(Common.LogonUser.LogonUserName, false);     //cleanup
		}

		#region Helper Methods

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		private string GetBceBtsMessage(Guid issuedUnitGuid, string reactions, bool breakARule)
		{
			string query = 
				"SELECT TOP 1 BU.BloodUnitGuid, P.VistaPatientId, P.PatientSsn, P.PatientLastName, P.PatientFirstName, " +
				"dbo.fnTimeZoneConversionDateTime('" + UnitTestConfig.TestDivisionCode + "', 0, IU.IssueDateTime) AS IssueDateTime " +
				"FROM dbo.BloodUnit BU " +
				"JOIN dbo.IssuedUnit IU ON BU.BloodUnitGuid = IU.BloodUnitGuid " +
				"JOIN dbo.BloodUnitStatusCodeCurrent BUSCC ON BU.BloodUnitGuid = BUSCC.BloodUnitGuid " +
				"JOIN dbo.OrderedUnit OU ON BU.BloodUnitGuid = OU.BloodUnitGuid " +
				"JOIN dbo.OrderedComponent OC ON OU.OrderedComponentGuid = OC.OrderedComponentGuid " +
				"JOIN dbo.PatientOrder PO ON OC.PatientOrderGuid = PO.PatientOrderGuid " +
				"JOIN dbo.PatientTreatment PT ON PO.PatientTreatmentGuid = PT.PatientTreatmentGuid " +
				"JOIN dbo.Patient P ON P.PatientGuid = PT.PatientGuid " +
				"WHERE UnitStatusCode = 'I' AND IU.IssuedUnitGuid = '" + issuedUnitGuid.ToString() + "'";
			// 
			DataRow drPatientAndBloodUnit = UnitTestUtil.RunSQLGetDataRow( query );
			Assert.IsNotNull( drPatientAndBloodUnit, "IssuedUnit Data Check" );
			//
			string bloodUnitGuid = drPatientAndBloodUnit["BloodUnitGuid"].ToString();
			string vistaPatientId = drPatientAndBloodUnit["VistaPatientId"].ToString();
			string patientSsn = drPatientAndBloodUnit["PatientSsn"].ToString();
			string patientLastName = drPatientAndBloodUnit["PatientLastName"].ToString();
			string patientFirstName = drPatientAndBloodUnit["PatientFirstName"].ToString();
			//
			DateTime issueDateTime = DateTime.Parse( Convert.ToDateTime(drPatientAndBloodUnit["IssueDateTime"]).ToString(Common.VBECSDateTime.DateTimeFormat) );
			DateTime transfusionStartDateTime = issueDateTime;
			DateTime transfusionEndDateTime = issueDateTime;
			//
			string issueDateTimeString = issueDateTime.ToString("yyyyMMddHHmmss");
			string transfusionStartDateTimeString = transfusionStartDateTime.ToString("yyyyMMddHHmmss");
			string transfusionEndDateTimeString = transfusionEndDateTime.ToString("yyyyMMddHHmmss");
			if (breakARule)
			{
				transfusionEndDateTimeString = issueDateTime.AddDays(-3).ToString("yyyyMMddHHmmss");
			}
			//
			return "MSH^~|\\&^WBC^500^VBECS^500^20110411084656^^BTS~O31^WBC201104110846564233^T^2.5^^^AL^AL^" + 
				CARRIAGE_RETURN +
				"PID^^^" + vistaPatientId + "^^" + patientLastName + "~" + patientFirstName + "~^^19151101^M^^^^^^^^^^^" + patientSsn + "^^^^^^^^^^^^^^^^^^^^" + 
				CARRIAGE_RETURN +
				"PV1^^O^CO-A/S DISPOSITION C~~~~~~~~~589^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" +
				CARRIAGE_RETURN +
				"ORC^SC^22167000~OR^^^^^^^^^^53333~USER~ONE^^^^^^^^^^^^^" +
				CARRIAGE_RETURN +
				"BPO^1^~RED BLOOD CELLS^^" +
				CARRIAGE_RETURN +
				"BTX^1^" + bloodUnitGuid + "^^^^^^^500^^TX^F^20110411084652^~ONE~USER^~TWO~USER^" + transfusionStartDateTimeString + "^" + transfusionEndDateTimeString + reactions +
				CARRIAGE_RETURN +
				"NTE^^^MAIN COMMENT HERE^RE" + 
				CARRIAGE_RETURN +
				"NTE^^^Testing Purposes^VA-BCR" +
				CARRIAGE_RETURN;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="length"></param>
		/// <returns></returns>
		private string GetLongComment(int length)
		{
			string output = string.Empty;
			//
			for (int idx = 0; idx < length; idx++)
			{
				output += "Z";
			}
			//
			return output;
		}

		#region Locking

        ///// <summary>
        ///// 
        ///// </summary>
        ///// <param name="lockErrorMessage"></param>
        ///// <param name="recordGuid"></param>
        ///// <returns></returns>
        //private static bool SetLocks(params System.Guid[] recordGuids)
        //{
        //    foreach (System.Guid recordGuid in recordGuids)
        //    {
        //        if ( !LockRecord(recordGuid) )
        //        {
        //            UnlockRecords(recordGuids, this._lockfor, Common.LockFunctions.UC069PostTransfusionInformation);
        //            //
        //            return false;
        //        }
        //    }
        //    //
        //    return true;
        //}

        ///// <summary>
        ///// 
        ///// </summary>
        ///// <param name="recordGuid"></param>
        ///// <returns></returns>
        //private static bool LockRecord(System.Guid recordGuid)
        //{
        //    if (recordGuid == System.Guid.Empty || IsRecordAlreadyLockedInUC(recordGuid))
        //    {
        //        return false;
        //    }
        //    //
        //    System.Data.DataTable dtLock = DAL.LockManager.LockUseCase(recordGuid, Common.LockFunctions.UC069PostTransfusionInformation, _lockFormId, Common.LockFunctions.UC069PostTransfusionInformation, false, _lockFormId);
        //    //
        //    if (dtLock.Rows.Count == 0 || dtLock.Rows.Count > 1)
        //    {
        //        return false;
        //    }
        //    //
        //    Lock tmpLock = new BOL.Lock(dtLock.Rows[0]);
        //    //
        //    // Check if there was a conflict
        //    if (tmpLock.LockConflictIndicator)
        //    {
        //        return (tmpLock.SessionGuid != Common.LogonUser.VbecsSessionGuid);
        //    }
        //    //
        //    return true;
        //}

        ///// <summary>
        ///// 
        ///// </summary>
        ///// <param name="recordGuid"></param>
        ///// <returns></returns>
        //private static bool IsRecordAlreadyLockedInUC(System.Guid recordGuid)
        //{
        //    BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid, false);

        //    if (tmpLock.LockedFormId)
        //    }
        //    //
        //    return(hasRecord);
        //}

		/// <summary>
		/// 
		/// </summary>
		private static void UnlockRecords(Guid [] recordGuids, int lockFormId, Common.LockFunctions lockFunction )
		{
            DAL.LockManager.UnlockRecords(recordGuids, lockFormId, lockFunction);
		}

        ///// <summary>
        ///// 
        ///// </summary>
        ///// <param name="tmpLock"></param>
        //private static bool CheckLockConflicts(Lock tmpLock)
        //{
        //    // Have to do ToUpper Cuz DB and Windows might not be EXACTLY da same
        //    if (tmpLock.UserNTLoginId.ToUpper() != Common.LogonUser.LogonUserName.ToUpper())
        //    {
        //        return false;
        //    }
        //    //
        //    if (tmpLock.LockFunctionId == Common.LockFunctions.UC069PostTransfusionInformation)
        //    {
        //        // For all NON UC_38 locks, if the session is different, it's a same record lock
        //        if (tmpLock.SessionGuid != Common.LogonUser.VbecsSessionGuid)
        //        {
        //            return false;
        //        }
        //    }
        //    else
        //    {
        //        return false;
        //    }
        //    //
        //    return true;
        //}

		#endregion 

		#endregion
	}
}
#endif